/* Explicación de programación para la generación de indicadores y medida de 
pobreza multidimensional:

El presente archivo genera los indicadores que se utilizan para identificar
y medir carencias en las dimensiones de Educación, Salud, Trabajo y Seguridad
Social, Vivienda y Entorno, y Redes y Cohesión Social. Dichas dimensiones son
agregadas para obtener el índice de pobreza multidimensional (variable 
pobreza_multi) de la metodología 2024 para la base de datos de la Encuesta
Casen 2024 publicada en la página del Observatorio Social del Ministerio de 
Desarrollo Social y Familia.

El do file se organiza de la siguiente forma:

Primero, en la línea 67 de la programación debe detallar la ubicación donde 
descargue la base de datos publicada en la página del Observatorio Social.
Adicionalmente, se deben descargar las bases SIMCE de 4º Básico y 2º Medio
del año 2024 a nivel de establecimiento, disponibles en el siguiente link, y 
guardarlas en el mismo directorio que se especifica más abajo (la apertura de
base de datos se realiza en las líneas 301 y 310).

Link bases SIMCE: https://informacionestadistica.agenciaeducacion.cl/#/bases


En el apartado "PREVIO" se excluyen los casos pertenecientes a núcleos
compuestos por el servicio doméstico puertas adentro (nucleo = 0), así como 
también se crean variables necesarias para poder elaborar las respectivas 
carencias en cada una de las dimensiones consideradas.

"EDUCACIÓN", "SALUD", "TRABAJO Y SEGURIDAD SOCIAL", "VIVIENDA Y ENTORNO" y 
"REDES Y COHESIÓN SOCIAL" elaboran cada una de las carencias respectivas a la 
dimensión respectiva. Dentro del apartado se encuentran las definiciones de 
población de referencia, así como también la carencia a nivel poblacional.

"AGREGACIÓN A NIVEL DE HOGARES" genera las variables agregadas a nivel de 
hogar para cada carencia, según sus poblaciones de referencia.

"INDICADOR SINTÉTICO" se compone de 2 etapas. Por un lado, el proceso de 
identificación, en donde se determinan cuáles hogares clasifican en situación
de pobreza multidimensional. Primero, se pondera a cada una de las 
carencias; de acuerdo a la actual metodología establecida para Chile, se 
considera que cada dimensión tradicional: Educación, Salud, Trabajo y Seguridad 
Social, Vivienda y Entorno, y Redes y Cohesión Social; tienen la misma 
ponderación (20%). Dentro de cada dimensión, los indicadores tienen igual peso. 
Posteriormente, se establece el vector de conteo, que corresponde a la suma 
ponderada de cada indicador. Además, se identifican los hogares que no cuentan 
con información en alguno de los indicadores que componen la medida. Una vez 
determinado el vector de conteo, se clasifica la situación de pobreza para los 
hogares que disponen de información completa. El umbral para determinar una 
situación de pobreza es de k >= 25% (carente en más de una dimensión), 
calculando así el indicador de Incidencia de Pobreza Multidimensional (“h_k”).

En la segunda etapa, la agregación, se crean el indicador de intensidad de 
la pobreza multidimensional (“a_k”), el cual establece el promedio ponderado de 
carencias que experimentan los hogares en situación de pobreza. Asimismo, en 
esta etapa se elabora la tasa de recuento ajustada “m_k”, que corresponde a la 
multiplicación de la incidencia (h_k) por la intensidad de la pobreza (a_k).

Finalmente, a modo de simplificar, la incidencia es renombrada como 
"pobreza_multi".
*/



clear all
set more off, permanently

global directorio "" //Agregar directorio 

*Carga de base de datos
use "$directorio\casen_2024.dta"


capture drop hh_d_* pobreza_multi*

********************************************************************************
********************************PREVIO******************************************
{/*****************************************************************************/

*Excluimos el nucleo 0:
drop if nucleo == 0

*Escolaridad considerando a menores de 15 años

capture drop esc_2
gen esc_2=esc
replace esc_2=0 if (esc==. & e6a<=5) & e6a!=-88
replace esc_2=e6b if (esc==. & e6a==7) & e6a!=-88
replace esc_2=e6b+8 if (esc==. & (e6a==9|e6a==11)) & e6a!=-88
replace esc_2=. if (e6a==-88|e6b==-88) & esc==.


*Generamos curso y grado para carencia de rezago

capture drop grado
gen grado=.
replace grado=1 if e6b==1 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=2 if e6b==2 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=3 if e6b==3 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=4 if e6b==4 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=5 if e6b==5 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=6 if e6b==6 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=7 if e6b==7 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=8 if e6b==8 & e6a==7 & e6a!=-88 & e6b!=-88
replace grado=9 if e6b==1 & (e6a==9|e6a==11) & e6a!=-88 & e6b!=-88
replace grado=10 if e6b==2 & (e6a==9|e6a==11) & e6a!=-88 & e6b! =-88
replace grado=11 if e6b==3 & (e6a==9|e6a==11) & e6a!=-88 & e6b!=-88
replace grado=12 if (e6b==4|e6b==5) & (e6a==9|e6a==11) & e6a!=-88 & e6b!=-88

capture drop curso
gen curso=grado+6
replace curso = 6 if e6a == 1 // para los que nunca asistieron

}


********************************************************************************
******************************EDUCACIÓN*****************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 1: Asistencia escolar

El hogar es carente en asistencia escolar si:
	• Al menos una persona integrante de 2 a 4 años no está asistiendo a un 
	establecimiento educacional por motivos no relacionados con las preferencias 
	de la familia (dificultad económica, no fue priorizado, no hay matrícula, no 
	lo aceptan, no lo reciben por discapacidad, dificultad de acceso); o,
	• Al menos una persona integrante de 5 a 18 años no asiste a un 
	establecimiento educacional y no ha egresado de cuarto medio; o,
	• Al menos una persona integrante de 6 a 26 años tiene una condición de 
	salud permanente y/o de larga duración y no asiste a un establecimiento 
	educacional.

*/
********************************************************************************

*Población de referencia
capture drop pob_asis d_asis
gen pob_asis=.
replace pob_asis=0 if (edad<2 | edad>18) & edad!=.
replace pob_asis=1 if (edad>=2 & edad<=18) & edad!=.
replace pob_asis=1 if (edad>=6 & edad<=26) & (s31_1 == 1 | s31_2 == 1 | s31_3 == 1 | s31_4 == 1 | s31_5 == 1 | s31_6 == 1)
replace pob_asis=0 if (edad<6) & (s31_1 == 1 | s31_2 == 1 | s31_3 == 1 | s31_4 == 1 | s31_5 == 1 | s31_6 == 1)
replace pob_asis=0 if (esc_2>=12 & esc_2!=.)
replace pob_asis=8 if esc_2==. & pob_asis==1 & edad>=15
replace pob_asis=8 if pob_asis==.
	
* Incidencia a nivel individual
capture drop d_asis
gen d_asis=.
replace d_asis=1 if (asiste==2) & pob_asis==1 & inrange(edad,2,4) & inlist(e4a,11,10,9,8,7,6)
replace d_asis=0 if (asiste==2) & pob_asis==1 & inrange(edad,2,4) & d_asis != 1
replace d_asis=0 if (asiste==1) & pob_asis==1 & inrange(edad,2,4)

replace d_asis=1 if (asiste==2) & pob_asis==1 & edad >= 5 
replace d_asis=0 if (asiste==1) & pob_asis==1 & edad >= 5 

replace d_asis=9 if (asiste==2) & pob_asis==1 & inrange(edad,2,4) & inlist(e4a,-88,-99)
replace d_asis=9 if d_asis==. & pob_asis==1 

********************************************************************************
/* Indicador 2: Rezago escolar 

El hogar es carente en rezago escolar si:
	• Al menos una persona integrante de 21 años o menos asiste a educación 
	básica o media, y se encuentra retrasado dos o más años respecto al curso 
	que le corresponde de acuerdo con su edad; o,
	• Al menos una persona integrante de 21 años o menos no asiste, no ha 
	completado su educación obligatoria y el último curso aprobado corresponde 
	a dos o más años de retraso respecto al curso que le corresponde por edad.


*/
********************************************************************************

*Población de referencia
capture drop pob_rez 

gen pob_rez=.
*Mayores a 21 no se consideran
replace pob_rez=0 if edad>21 & edad!=.
*Quienes asisten distinto de basica o media no se consideran
replace pob_rez=0 if (e6a_asiste!=7 & e6a_asiste!=9 & e6a_asiste!=11) & (edad<=21)
*Asiste a basica o media y es menor de 21 se considera
replace pob_rez=1 if (asiste==1 & inlist(e6a_asiste,7,9,11) & (edad<=21))
*Escolaridad de 12 o mas (completaron media) no se consideran
replace pob_rez=0 if esc_2>=12 & esc_2!=. & pob_rez==1
*no asisten, ultimo nivel fue basica, nunca asistieron, o ultimo nivel fue media y no lo completaron (hasta curso igual a 3) se considera
replace pob_rez=1 if asiste==2 & edad<=21 & ((inlist(e6a_no_asiste,1) & inrange(edad,6,21)) | inlist(e6a_no_asiste,7) | (inlist(e6a_no_asiste,9,11) & inlist(e6b_no_asiste,1,2,3)))
replace pob_rez = 8 if asiste == 2 & inlist(e6b_no_asiste,-88,-99) & (edad<=21) & inlist(e6a_no_asiste,7,9,11)

* Privación a nivel individual
capture drop d_rez
gen d_rez=.
replace d_rez=1 if(edad-curso>=2) & pob_rez==1 & asiste!=. & esc_2!=. & edad!=.
replace d_rez=0 if(edad-curso<2) & pob_rez==1 & asiste!=. & esc_2!=. & edad!=.
replace d_rez=9 if d_rez==. & pob_rez==1

********************************************************************************
/* Indicador 3: Escolaridad

El hogar es carente en escolaridad si:
	• Al menos una persona integrante de 19 a 21 años tiene menos años de 
	escolaridad que los establecidos por ley, de acuerdo con su edad y no está 
	estudiando; o,
	• Al menos una persona integrante de 2119 o más años y menor de 65 años 
	tiene menos años de escolaridad que los establecidos por ley, de acuerdo 
	con su edad; o,
	• Al menos una persona integrante de 65 años o más es analfabeto 
	(no sabe leer y escribir).
 

*/
********************************************************************************
/*
Se definen exigencias por ley para cada cohorte que se ve afectada por los 
cambios legislativos:

a. 1920 a 1929: 4 años
b. 1930 a 1965: 6 años
c. 1966 a 2002: 8 años
d. 2003 en adelante: 12 años
*/

capture drop tedad
*A quienes les comenzó a regir los 8 años de escolaridad obligatoria en el 2022 tienen 71 años (1967)
gen tedad=1 if (edad>39 & edad<65)
*A quienes les comenzó a regir los 12 años de escolaridad obligatoria en el 2022 tienen 39 años (2003)
replace tedad=2 if (edad>18 & edad<=39)

* Población de referencia
capture drop pob_esc
gen pob_esc=.
replace pob_esc=1 if (edad>18) & edad!=.
replace pob_esc=0 if (edad<=18) & edad!=.
replace pob_esc = 0 if edad > 18 & edad != . & e6a == 5
replace pob_esc=8 if pob_esc==.
	
* Privación a nivel individual
capture drop d_esc
gen d_esc=.
replace d_esc=0 if ((tedad==1 & esc>=8)|(tedad==2 & esc>=12)) & esc!=. & pob_esc==1 & inrange(edad,18,64)
replace d_esc=1 if ((tedad==1 & esc<8)|(tedad==2 & esc<12 )) & esc!=. & pob_esc==1  & inrange(edad,18,64)
replace d_esc = 0 if edad >= 65 & inlist(e1,1) & pob_esc==1 
replace d_esc = 1 if edad >= 65 & e1 != 1 & pob_esc==1 

*Rescate poblacion de 19 a 21 que se encuentra asistiendo 
replace d_esc = 0 if inrange(edad,19,21) & esc < 12 & asiste == 1 & d_esc == 1 & pob_esc == 1
replace d_esc = 9 if esc == . & pob_esc==1 & inrange(edad,18,64) //missing quienes no tienen escolaridad en el tramo que se evalua
replace d_esc=9 if d_esc==. & pob_esc==1

capture drop pob_escolaridad
gen pob_escolaridad=.
replace pob_escolaridad=0 if edad <=18
replace pob_escolaridad=1 if edad>18 & edad!=. & e6a!=5
 

capture drop tedad
 
gen tedad=1 if (edad>=65 & edad!=.)
 
replace tedad=2 if (edad>39 & edad<65)
* A quienes les comenzó a regir los 12 años de escolaridad obligatoria en el 2024 tienen 39 años (2003)
replace tedad=3 if (edad>18 & edad<=39)
 
 
capture drop d_escolaridad
gen d_escolaridad=.
replace d_escolaridad=1 if tedad==3 & esc<12				 	& pob_escolaridad==1   
replace d_escolaridad=1 if tedad==2 & esc<8  					& pob_escolaridad==1
replace d_escolaridad=1 if tedad==1 & inlist(e1,2,3,4)			& pob_escolaridad==1
 
replace d_escolaridad=0 if ((tedad==3 & esc>=12)|(tedad==2 & esc>=8)|(tedad==1 & inlist(e1,1))) & esc!=-88 & pob_escolaridad==1
replace d_escolaridad=0 if inrange(edad, 19,21) & esc<12 & inlist(e3,1) & pob_escolaridad==1 
replace d_escolaridad=0 if inlist(tedad,1) & inlist(e1,1) & pob_escolaridad==1 
replace d_escolaridad=9 if esc== . & pob_escolaridad==1 & (edad>18 & edad<65)


********************************************************************************
/* Indicador 4: Aprendizaje escolar en el establecimiento

El hogar es carente en aprendizaje escolar en el establecimiento si:
	• Al menos una persona integrante en educación básica o media asiste a un 
	establecimiento donde más del 55% de sus estudiantes son clasificados en un 
	nivel de aprendizaje insuficiente en matemática y lenguaje, según los 
	estándares de aprendizaje vigentes en la última prueba SIMCE.


*/
********************************************************************************

*Poblacion de referencia
capture drop pob_ape
gen pob_ape = 0
replace pob_ape = 1 if inrange(edad,5,21) & inlist(e6a_asiste,7,9,11)

capture frame drop simcebas
frame create simcebas
frame change simcebas 

import excel "$directorio\simce4b2024_rbd_preliminar.xlsx", sheet("Hoja1") firstrow clear
keep rbd palu_eda_ins_lect4b_rbd palu_eda_ins_mate4b_rbd
rename rbd e9rbd_esc
tempfile bas
save `bas'

capture frame drop simcemed
frame create simcemed
frame change simcemed 
import excel "$directorio\simce2m2024_rbd_preliminar.xlsx", sheet("Hoja1") firstrow clear
keep rbd palu_eda_ins_lect2m_rbd palu_eda_ins_mate2m_rbd 


rename rbd e9rbd_esc
tempfile medi
save `medi'

frame change default
merge m:1 e9rbd_esc using `bas', gen(simce_bas)
drop if simce_bas == 2

merge m:1 e9rbd_esc using `medi', gen(simce_med)
drop if simce_med == 2

*Basica
capture drop basica
gen basica = 0 if pob_ape == 1 & e6a_asiste == 7
replace basica = 1 if ((palu_eda_ins_mate4b_rbd > 55 & palu_eda_ins_mate4b_rbd != .) & (palu_eda_ins_lect4b_rbd > 55 & palu_eda_ins_lect4b_rbd != .)) & e6a_asiste == 7
replace basica = 9 if basica == 0 & ((palu_eda_ins_mate4b_rbd == . & palu_eda_ins_lect4b_rbd > 55 & palu_eda_ins_lect4b_rbd != .) | (palu_eda_ins_lect4b_rbd == . & palu_eda_ins_mate4b_rbd > 55  & palu_eda_ins_mate4b_rbd != .)) & e6a_asiste == 7
replace basica = 9 if basica == 0 & (palu_eda_ins_mate4b_rbd == . & palu_eda_ins_lect4b_rbd == .) & e6a_asiste == 7

*Media
capture drop media
gen media = 0 if pob_ape == 1 & inlist(e6a_asiste,9,11)
replace media = 1 if ((palu_eda_ins_mate2m_rbd > 55 & palu_eda_ins_mate2m_rbd != .) & (palu_eda_ins_lect2m_rbd > 55 & palu_eda_ins_lect2m_rbd != . )) & inlist(e6a_asiste,9,11)
replace media = 9 if media == 0 & ((palu_eda_ins_mate2m_rbd == . & palu_eda_ins_lect2m_rbd > 55 & palu_eda_ins_lect2m_rbd != .) | (palu_eda_ins_lect2m_rbd == . & palu_eda_ins_mate2m_rbd > 55 & palu_eda_ins_mate2m_rbd != .)) & inlist(e6a_asiste,9,11)
replace media = 9 if media == 0 & (palu_eda_ins_mate2m_rbd == . & palu_eda_ins_lect2m_rbd == .) & inlist(e6a_asiste,9,11)

capture drop d_ape
gen d_ape = 0 if (basica == 0) & pob_ape == 1 & e6a_asiste == 7
replace d_ape = 1 if (basica == 1) & pob_ape == 1 & e6a_asiste == 7
replace d_ape = 0 if (media == 0) & pob_ape == 1 & inlist(e6a_asiste,9,11)
replace d_ape = 1 if (media == 1) & pob_ape == 1 & inlist(e6a_asiste,9,11)

replace d_ape = 9 if d_ape == . & pob_ape == 1

}


********************************************************************************
********************************SALUD*******************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 5: Atención de salud

El hogar es carente en atención de salud si:
	• Al menos una persona integrante tuvo un problema de salud en los últimos 
	3 meses y no tuvo consulta ni atención por alguna de las siguientes razones: 
	pensó en consultar, pero no tuvo tiempo; pensó en consultar, pero no tuvo 
	dinero; pensó en consultar, pero le cuesta mucho llegar al lugar de 
	atención; pidió hora, pero no la obtuvo; la consulta o atención fue 
	postergada o cancelada; o,
	• Al menos una persona integrante tuvo un problema de salud en los últimos 
	3 meses, accedió a consulta o atención, pero tuvo 3 o más de los siguientes 
	problemas: problemas para llegar; para conseguir hora; para ser atendido; 
	para pagar la atención; o para acceder a medicamentos; o,
	• Al menos una persona integrante tuvo un problema de salud en los últimos 
	3 meses, accedió a consulta o atención, pero tuvo problemas para acceder a 
	medicamentos.

*/
********************************************************************************

*Poblacion de referencia
capture drop pob_acc
gen pob_acc=(inrange(s16,1,4))
replace pob_acc=8 if inlist(s16,-88,-99)

*Cálculo carencia
capture drop d_acc
*gente que no tuvo atencion

gen d_acc = 0 if pob_acc == 1
replace d_acc = 1 if inlist(s18,8,9,10,11,15) & pob_acc == 1

*Gente que tuvo atencion
foreach i in a b c d{
	capture drop raz_`i' 
	gen raz_`i' = 0 if s19`i' != .
	replace raz_`i' = 1 if s19`i' == 1
	replace raz_`i' = 0 if s19`i' == 2
	capture drop  nsraz_`i'
	gen nsraz_`i' = 1 if inlist(s19`i',-88,-99)
}
capture drop cont
egen cont = rowtotal(raz_a raz_b raz_c raz_d) if s17 == 1
capture drop contns
egen contns = rowtotal(nsraz_a nsraz_b nsraz_c nsraz_d) if s17 == 1
replace d_acc = 1 if cont >= 3 & cont != . & pob_acc == 1

*Dejamos como carente, independiente del conteo previo, a quienes no tienen acceso a medicamentos
replace d_acc = 1 if s19e == 1

*Ajuste a 9 para los casos que solo responden combinatia entre No y NSNR
replace d_acc = 9 if inlist(s18,-88,-99) & d_acc == 0
replace d_acc = 9 if inlist(s17,-88,-99) & d_acc == 0
replace d_acc = 9 if cont == 0 & contns >= 3 & contns != . & s17 == 1 & d_acc == 0
replace d_acc = 9 if cont == 1 & contns >= 2 & contns != . & s17 == 1 & d_acc == 0
replace d_acc = 9 if cont == 2 & contns >= 1 & contns != . & s17 == 1 & d_acc == 0
replace d_acc = 9 if inlist(s19e,-88,-99) & d_acc == 0
replace d_acc = 9 if s17b == -88 & cont == 2 & d_acc == 0


********************************************************************************
/* Indicador 6: Acceso a alimentos

Un hogar es carente en acceso a alimentos si:
	• Alguna persona integrante experimenta, en los últimos 12 meses, 4 
	situaciones o más relativas a dificultades de acceso a una alimentación 
	adecuada por falta de dinero o recursos, según la escala de Experiencia 
	de Inseguridad Alimentaria de la FAO.

*/
********************************************************************************

*Poblacion de referencia
capture drop pob_ali
gen pob_ali = (numper != .)
replace pob_ali = 8 if numper == .

generate preocupa = r8a
generate saludable = r8b
generate variedad = r8c
generate deja_des = r8d
generate comio_menos = r8e
generate sin_alim = r8f
generate hambre = r8g
generate todo_dia = r8h

foreach j in preocupa saludable variedad deja_des comio_menos sin_alim hambre todo_dia{
		
	replace `j' = 0 if `j' == 2
	label variable `j' "pregunta IA `j'"
	label define `j' 0 "No" 1 "Si", modify
	label values `j' `j'
	
}

capture drop puntia_todo
egen puntia_todo = rowtotal(preocupa saludable variedad deja_des comio_menos sin_alim hambre todo_dia)

*Carencia
capture drop d_ia

gen d_ali = 0
replace d_ali = 1 if puntia_todo >= 4 & puntia_todo != . & pob_ali == 1

********************************************************************************
/* Indicador 7: Acceso a salud preventiva

Un hogar es carente en acceso a salud preventiva si:
	• Al menos una persona integrante entre 0 a 9 años, no haya asistido durante 
	los últimos 12 meses a un control de salud de niño y niña sano.


*/
********************************************************************************

*Poblacion de referencia
capture drop pob_contprev
gen pob_contprev = 0
replace pob_contprev = 1 if inrange(edad,0,9)

*Carencia
capture drop d_contprev
gen d_contprev = 0 if pob_contprev == 1
replace d_contprev = 1 if (s26b_1 == 0 | s26a == 0) & pob_contprev == 1
replace d_contprev = 9 if s26a == -88 & pob_contprev == 1

********************************************************************************
/* Indicador 8: Apoyo en cuidado de personas con dependencia funcional

Un hogar es carente en apoyo en cuidado de personas con dependencia funcional si:
	• Al menos una persona integrante con dependencia funcional moderada o 
	severa, no cuenta con ayuda o apoyo externo al hogar para realizar sus 
	actividades diarias; o, 
	• Al menos una persona integrante con dependencia funcional leve no cuenta 
	con una persona interna ni externa al hogar que ayude o apoye en 
	la realización de sus actividades diarias.

*/
********************************************************************************

*Poblacion de referencia
capture drop pob_dpf 
gen pob_dpf = 1 if inlist(depen_grado,1,2,3)

*Cálculo de la carencia
capture drop d_dpf
gen d_dpf=.
replace d_dpf=1 if inlist(depen_grado,1,2)	& inlist(s34c,2) & pob_dpf==1
replace d_dpf=0 if inlist(depen_grado,1,2)	& inlist(s34c,1) & pob_dpf==1
replace d_dpf=0 if inlist(depen_grado,3) & (inlist(s34c,1) | inlist(s34a,1))
replace d_dpf=1 if inlist(depen_grado,3) & inlist(s34c,2) & inlist(s34a,2)

}


********************************************************************************
*******************************TRABAJO******************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 9: Ocupación y subempleo

Un hogar es carente en ocupación y subempleo si:
	• Al menos una persona integrante mayor de 18 años (o menor de 19 años si 
	ha completado la enseñanza media), está desempleada; o, 
	• Al menos una persona integrante mayor de 18 años (o menor de 19 años si ha 
	completado la enseñanza media), está subempleada por insuficiencia de horas, 
	entendiéndose esto como personas que trabajan 30 horas o menos, y que desean 
	y están disponibles para trabajar más horas a la semana de forma inmediata.

*/
********************************************************************************

*Poblacion de referencia
capture drop pob_actsub
gen pob_actsub=.
replace pob_actsub=1 if edad>18
replace pob_actsub=0 if edad<=18
*Identificamos a quienes completaron educacion media
replace pob_actsub = 1 if ((inlist(e6a,12,13,14,15)) | (e6a_no_asiste == 9 & e6b_no_asiste == 4) | (e6a_no_asiste == 11 & inlist(e6b_no_asiste,4,5))) & inrange(edad,15,18)
*dejamos fuera a quienes no tienen informacion en escolaridad
replace pob_actsub=8 if esc==. & (edad>14 & edad<=18)

* Privación a nivel individual

capture drop subempleo
g subempleo=.
replace subempleo=1 if o10<=30 & o11==1 & o10 != -88 & activ == 1
replace subempleo=0 if o10<=30 & inlist(o11,2,3)  & o10 != -88  & activ == 1
replace subempleo=0 if o10>30 & o10<999 & o10 != -88  & activ == 1
replace subempleo = 9 if o10 == -88 | (o10<=30 & o10 != -88 & o11 == -88)  & activ == 1
replace subempleo = 9 if o10 == -88 & o11 != -88  & activ == 1

capture drop d_actsub
gen d_actsub=0 if pob_actsub == 1
*Identificamos a los desocupados
replace d_actsub=1 if (activ==2) & pob_actsub==1
*Identificamos a los subempleados
replace d_actsub = 1 if subempleo == 1  & pob_actsub==1
*Quienes no sabemos la cantidad de horas y no sabemos si estan dispuestos a trabajar mas los clasificamos como 9
replace d_actsub = 9 if subempleo == 9  & pob_actsub==1


********************************************************************************
/* Indicador 10: Informalidad

Un hogar es carente en informalidad si al menos una persona de 15 años o más se 
encuentra trabajando en el sector informal o en una ocupación informal. Para definir 
la informalidad se consideran los siguientes criterios para los ocupados:
	• Es independiente —empleador o por cuenta propia— y: 
		- No inicia actividades en el Servicio de Impuestos Internos (SII). 
		- En caso de no contar con información respecto al inicio de actividades 
		se considera carente si su oficio es no calificado.
	• Es dependiente y: 
		- Emite boleta de honorarios; o, 
		- No se encuentra adscrito en un sistema previsional de salud o no cotiza 
		en el sistema previsional (sistema de pensiones).
	• Es familiar no remunerado.



*/
********************************************************************************
*Poblacion de referencia
capture drop pob_inf

gen pob_inf = 0
replace pob_inf = 1 if edad >= 15 & activ == 1

*Calculo de la carencia
capture drop d_inf

*DEPENDIENTES
*Son informales todos los dependientes que emiten boletas de honorarios
gen d_inf = 1 if pob_inf == 1 & inrange(o15,3,8) & o14 == 1
*Quienes no emiten boleta de honorarios y son dependientes, son informales quienes o no cotizan afp o en salud:
replace d_inf = 1 if (pob_inf == 1) & (inrange(o15,3,8) & o14 != 1) & (s13 == 4 | (o31 == 2 | o32 == 6))

*Son formales entonces quienes no emiten boleta de honorarios y cotizan en afp y sistema de salud:
replace d_inf = 0 if (pob_inf == 1) & (inrange(o15,3,8) & o14 != 1 & o14 != -88) & (s13 != 4 & s13 != -88 & ((o31 == 1 & (o32 != 6 & o32 != -88))))

*Todo el resto de dependientes son missing:
replace d_inf = 9 if d_inf == . & (pob_inf == 1) & inrange(o15,3,8)

*FAMILIAR NO REMUNERADO:
*Familiares no remunerados son informales:
replace d_inf = 1 if (pob_inf == 1) & o15 == 9


*INDEPENDIENTES
*Son formales los independientes que inicien actividades de SII e informales quienes no:
replace d_inf = 0 if (pob_inf == 1) & inlist(o15,1,2) & o16 == 1
replace d_inf = 1 if (pob_inf == 1) & inlist(o15,1,2) & o16 == 2

*Recuperamos por oficio:
replace d_inf = 0 if (pob_inf == 1) & inlist(o15,1,2) & inlist(oficio1_08,1,2,3) & d_inf == .
replace d_inf = 1 if (pob_inf == 1) & inlist(o15,1,2) & inrange(oficio1_08,4,9) & d_inf == .

*Todos los independientes no identificados a esta altura son missing:
replace d_inf = 9 if d_inf == . & (pob_inf == 1) & inrange(o15,1,2)

******************************************************************************
/* Indicador 11: Jubilación

Un hogar es carente en jubilación si:
	• Al menos una persona en edad de jubilarse (mujeres de 60 años o más y 
	hombres de 65 años o más) no percibe pensión contributiva o no contributiva, 
	y no recibe otros ingresos por arriendos, retiro de utilidades, dividendos 
	e intereses.

*/
********************************************************************************

*Poblacion de referencia
gen pob_jub = .
replace pob_jub = 1 if edad > 64 & sexo == 1
replace pob_jub = 0 if edad <= 64 & sexo == 1
replace pob_jub = 1 if edad >59 & sexo == 2
replace pob_jub = 0 if edad <=59 & sexo == 2
replace pob_jub = 8 if pob_jub==.

gen d_jub = 0 if pob_jub == 1

egen otrospriv = rowtotal(yah1 yah2 yrut yre1 yama yre2 yre3),m
	
* Ingresos pensiones
egen ypenst = rowtotal(yjub_pgu02 yjub_aps02 yjub_conc yinv_aps02 yinv_con ymon_pgu02 ymon_con yorf yotp yjub_pgu03 yjub_aps03 yjub_pgu01 yinv_aps03 yinv_pbs ymon_pgu03 yesp_pgu02 yesp_pgu03 yesp),m


gen ing_jub=. 
replace ing_jub=1 if otrospriv>0 & otrospriv!=. & otrospriv!=.m & otrospriv!=.z
replace ing_jub=1 if ypenst>0 & ypenst!=. & ypenst!=.m & ypenst!=.z 
replace ing_jub=0 if ing_jub==.
label define ing_jub 1 "Tiene Ingresos para la Jubilación" 0 "No Tiene Ingresos para la Jubilación"
label values ing_jub ing_jub

replace d_jub = 1 if ing_jub == 0 & pob_jub == 1


********************************************************************************
/*
Indicador 12: Participación por cuidados

Un hogar es carente en participación por cuidados si:
	• Al menos una persona integrante no ocupada no ha buscado trabajo durante 
	el último mes por razones de cuidado; o,
	• Al menos una persona integrante no estudia por cuidar de otra persona 
	(incluye cuidado de alguien o por maternidad/paternidad).

*/
********************************************************************************

*Población de referencia
capture drop pob_cui 
gen pob_cui = 0
replace pob_cui = 1 if edad >= 15

*Carencia
capture drop d_cui
gen d_cui = . if pob_cui == 1
replace d_cui = 0 if (!inlist(o7,3,4,5) | !inlist(e5a,2,3)) & pob_cui == 1
*Inactivos que no trabajan por razones de cuidado
replace d_cui = 1 if inlist(o7,3,4,5) & pob_cui == 1
*Personas que no asisten a est. educ. por razones de cuidado
replace d_cui = 1 if inlist(e5a,2,3) & pob_cui == 1
*Quienes NS/NR en razones de asistencia y no aplican para razon de inactividad quedan como 9
replace d_cui = 9 if asiste == 2 & inlist(e5a,-88,-99) & d_cui == 0 & pob_cui == 1

}


********************************************************************************
*******************************VIVIENDA*****************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 13: Déficit cuantitativo

Un hogar es carente en déficit cuantitativo si presenta al menos alguno de los 
siguientes requerimientos: 
	• estado irrecuperable de la vivienda; 
	• existencia de hogares allegados;
	• existencia de núcleos familiares allegados hacinados; o,
	• existencia de hogares no allegados pero hacinados en viviendas no 
	ampliables.

Indicador 14: Déficit cualitativo:

Un hogar es carente en déficit cualitativo si tiene una vivienda que posee
alguno de los siguientes requerimientos:
	• requerimiento de ampliación por razones de hacinamiento; 
	• requerimiento de mejoramiento material y conservación; o, 
	• requerimiento por acceso a servicios básicos deficitarios.



*/
********************************************************************************

*Población de referencia
capture drop pob_defcuali pob_defcuanti

gen pob_defcuali =  (numper != .)
gen pob_defcuanti =  (numper != .)


*Calculo de la carencia
capture drop id_hogar
gen double id_hogar=folio
format id_hogar %21.0g

gen aux_nucleo=1 if pco2==1 & nucleo!=0
bys id_hogar: egen n_nucleo_hog=sum(aux_nucleo)
label var n_nucleo_hog "n total de nucleos en el hogar"
capture drop idhog_prin
gen	idhog_prin	=v28
label variable idhog_prin "Hogar principal de la vivienda"
label define idhog_prin 1 "Sí" 2 "No" 3 "ns/nr"
label values idhog_prin idhog_prin

capture drop nhog_viv
gen	nhog_viv = .
replace nhog_viv = 1 	if 	p9==1
replace nhog_viv = 1 	if p10==1
replace nhog_viv = p11 	if p10==2
label variable nhog_viv "Cantidad de hogares en la vivienda"

capture drop hog_prin
gen 	hog_prin=.
replace hog_prin=1  if (nhog_viv==1)
replace hog_prin=1  if (nhog_viv>1) & (idhog_prin==1)
replace hog_prin=2  if (nhog_viv>1) & (idhog_prin==2)
replace hog_prin=99 if (nhog_viv>1) & (idhog_prin==9)
replace hog_prin=99 if (nhog_viv==99)
replace hog_prin=. 	if hog_prin==99

label var hog_prin "Condición de ocupante principal del hogar"
label define hog_prin 1 "hogar principal" 2 "hogar no principal" 99 "sin información", modify
label values hog_prin hog_prin

*Número de hogares allegados
capture drop n_hog_allega
gen n_hog_allega =.
replace n_hog_allega=0 if (nhog_viv==1) & (hog_prin==1)
replace n_hog_allega=nhog_viv-1 if (nhog_viv>1) & (hog_prin==1)
replace n_hog_allega=0 if (nhog_viv>1) & (hog_prin>=2)
label var n_hog_allega "número de hogares allegados"
label values n_hog_allega n_hog_allega 

*Núcleo principal
gen nuc_prin=2
replace nuc_prin=1 if pco1==1 & pco2==1
replace nuc_prin=. if pco1==15
label var nuc_prin "condición de núcleo ocupante principal del hogar"
label define nuc_prin 1 "núcleo principal" 2 "núcleo no principal"
label values nuc_prin nuc_prin 

gen	con_muro = v3
gen	con_piso = v5
gen	con_techo = v7

label variable con_muro "Estado de conservación de los muros"
label define con_muro 1 "Bueno" 2 "Aceptable" 3 "Malo"
label variable con_piso "Estado de conservación del piso"
label define con_piso 1 "Bueno" 2 "Aceptable" 3 "Malo"
label variable con_techo "Estado de conservación del techo"
label define con_techo 1 "Bueno" 2 "Aceptable" 3 "Malo"

capture drop ind_estado
gen 	ind_estado=.
replace ind_estado=1 	if (con_muro==1) & (con_piso<=2) & (con_techo<=2)
replace ind_estado=2 	if (con_muro==2) & (con_piso<=2) & (con_techo<=2)
replace ind_estado=3 	if (con_muro==3) | (con_piso==3) | (con_techo==3)
replace ind_estado=99 	if (con_muro==9) | (con_piso==9) | (con_techo==9)

label var ind_estado "indice estado de la vivienda"
label define ind_estado 1 "bueno" 2 "regular" 3 "malo" 99 "ns/nr", modify
label values ind_estado ind_estado 

capture drop ind_tip
recode v1 (1/6 8 9=1)(7 10=2), generate(ind_tip)
	
label variable ind_tip "Tipo de vivienda"
label define ind_tip 1 "Aceptable" 2 "Irrecuperable", modify
label values ind_tip ind_tip

recode v2 (1/3=1)(4/5=2)(6=3)(9=99), generate(mat_muros)
recode v4 (1/3=1)(4/5=2)(6=3)(9=99), generate(mat_piso)
recode v6 (1/4=1)(5/6=2)(7/8=3)(9=99), generate(mat_techo)

label variable mat_muros "Materialidad Muro"
label define mat_muros 1 "Aceptable" 2 "Recuperable" 3 "Irrecuperable"
label values mat_muros mat_muros
label variable mat_piso "Materialidad Piso"
label define mat_piso 1 "Aceptable" 2 "Recuperable" 3 "Irrecuperable"
label values mat_piso mat_piso
label variable mat_techo "Materialidad Techo"
label define mat_techo 1 "Aceptable" 2 "Recuperable" 3 "Irrecuperable"
label values mat_techo mat_techo

capture drop ind_mat
gen 	ind_mat=.
replace ind_mat=1 if (mat_muros==1) & (mat_piso==1) & (mat_techo==1)
replace ind_mat=2 if (mat_muros==2) & (mat_piso==1) & (mat_techo==1)
replace ind_mat=3 if (mat_muros<=2) & (mat_piso==2) & (mat_techo==1)
replace ind_mat=3 if (mat_muros<=2) & (mat_piso==1) & (mat_techo==2)
replace ind_mat=3 if (mat_muros<=2) & (mat_piso==2) & (mat_techo==2)
replace ind_mat=4 if (mat_muros==3) | (mat_piso==3) | (mat_techo==3)
replace ind_mat=99 if (mat_muros==99) | (mat_piso==99) | (mat_techo==99)

capture drop dis_agua
recode v22 (1=1)(2/3=2)(9=.), generate(dis_agua)
label variable dis_agua "Distribución del agua"
label define dis_agua 1 "Aceptable", modify
label define dis_agua 2 "Deficitario", modify
label define dis_agua 99 "ns/nr", modify
label values dis_agua dis_agua


capture drop serv_higienico
recode v23_sistema (1/2=1) (3 4 7=2) (9=99), generate(serv_higienico)
replace serv_higienico=2 if v23==2
replace serv_higienico=99 if v23==.
	
label variable serv_higienico "Sistema de eliminación de escretas"
label define serv_higienico 1 "Aceptable" 2 "Deficitario", modify
label values serv_higienico serv_higienico

capture drop ori_aguarur ori_aguaurb
recode v20 (1 4/6=1) (7=2) (.=99), generate(ori_aguarur)
label variable ori_aguarur "Origen del agua rural"
label define ori_aguarur 1 "Aceptable" 2 "Deficitario", modify
label values ori_aguarur ori_aguarur

recode v20 (1=1) (4/7=2) (.=99), generate(ori_aguaurb)
label variable ori_aguaurb "Origen del agua urbano"
label define ori_aguaurb 1 "Aceptable" 2 "Deficitario", modify
label values ori_aguaurb ori_aguaurb

capture drop orig_agua
gen 	orig_agua=.
replace orig_agua=1  if (area==1) & (ori_aguaurb==1)
replace orig_agua=2  if (area==1) & (ori_aguaurb==2)
replace orig_agua=99 if (area==1) & (ori_aguaurb==99)
replace orig_agua=1  if (area==2) & (ori_aguarur==1)
replace orig_agua=2  if (area==2) & (ori_aguarur==2)
replace orig_agua=99 if (area==2) & (ori_aguarur==99)

label var orig_agua "origen del agua"
label define orig_agua 1 "aceptable" 2 "deficitario" 99 "ns/nr", modify
label values orig_agua orig_agua

capture drop ind_san
gen 	ind_san=.
replace ind_san=1 	if (dis_agua==1)  & (serv_higienico==1)  & (orig_agua==1)
replace ind_san=2 	if (dis_agua==2)  | (serv_higienico==2)  | (orig_agua==2)
replace ind_san=99 	if (dis_agua==99) | (serv_higienico==99) | (orig_agua==99)
replace ind_san=. if !inlist(ind_san,1,2)

label var ind_san "índice de saneamiento en la vivienda (acceso a servicios básicos)"
label define ind_san 1 "aceptable", modify
label define ind_san 2 "deficitario", modify
label define ind_san 99 "ns/nr", modify
label values ind_san ind_san

capture drop ind_cal_glob
gen 	ind_cal_glob=.
replace ind_cal_glob=1 	if (ind_tip==1)  & 	(ind_mat<=2)  & (ind_san==1)
replace ind_cal_glob=2 	if (ind_tip==1)  & 	(ind_mat<=3)  & (ind_san==2)
replace ind_cal_glob=2 	if (ind_tip==1)  & 	(ind_mat==3)  & (ind_san==1)
replace ind_cal_glob=3 	if ((ind_tip==2) |	(ind_mat==4)) & (ind_san<=2)
replace ind_cal_glob=99 if 					(ind_mat==99) | (ind_san==99)

label var ind_cal_glob "indice de calidad global de la vivienda"
label define ind_cal_glob 1 "aceptable" 2 "recuperable" 3 "irrecuperable" 99 "ns/nr", modify
label values ind_cal_glob ind_cal_glob

capture drop allega_int

gen allega_int=.
replace allega_int=0 if (n_nucleo_hog==1)
replace allega_int=1 if (n_nucleo_hog>=2)

label var allega_int "presencia de allegamiento interno en el hogar"
label define allega_int 0 "sin allegamiento interno", modify
label define allega_int 1 "con allegamiento interno", modify
label values allega_int allega_int

capture drop ndor_viv ndor_hog
gen	ndor_viv = v27a
label variable ndor_viv "Cantidad de dormitorios (uso exclusivo) en la vivienda"
	
gen	ndor_hog = v29a
label variable ndor_hog "Cantidad de dormitorios (uso exclusivo) que ocupa el hogar en la vivienda"

capture drop hacina
gen 	hacina=.
replace hacina=numper/ndor_viv 	if nhog_viv==1 & ndor_viv>0 & ndor_viv<99
replace hacina=8888 			if nhog_viv==1 & ndor_viv==0
replace hacina=9999 			if nhog_viv==1 & (ndor_viv==-88 | ndor_viv==-88)
replace hacina=numper/ndor_hog 	if nhog_viv>1 & ndor_hog>0 & ndor_hog<99
replace hacina=9999 			if nhog_viv>1 & (ndor_hog==-88 | ndor_hog==-88)
replace hacina=8888 			if nhog_viv>1 & ndor_hog==0
label var hacina "calculo hacinamiento"
label values hacina hacina


*ind_hacina=f(hacina)
capture drop ind_hacina
recode hacina (0/2.49=1)(2.5/3.49=2)(3.5/4.99=3)(5/8888=4)(9999=99),gen(ind_hacina)

label var ind_hacina "índice de hacinamiento"
label define ind_hacina 																				///
 1 "sin hacinamiento (menos de 2,5 personas por dormitorio)"											///
 2 "hacinamiento medio (entre 2,5 y 3,49 personas por dormitorio)"										///
 3 "hacinamiento alto (entre 3,5 y 4,99 personas por dormitorio)"										///
 4 "hacinamiento crítico (5 y más personas por dormitorio u hogar sin dormitorios de uso exclusivo)"	///
99 "sin dato (ns/nr en n° dormitorios de uso exclusivo)", modify
label values ind_hacina ind_hacina

******************************************
*********Requerimientos de mejora*********
******************************************

gen req_mejora=.
replace req_mejora=0 if (pco1==1) & (hog_prin==1) & (ind_estado<=2) & (ind_cal_glob==1)
replace req_mejora=1 if (pco1==1) & (hog_prin==1) & (ind_estado<=2) & (ind_cal_glob==2)
replace req_mejora=1 if (pco1==1) & (hog_prin==1) & (ind_estado==3) & (ind_cal_glob<=2)

label var req_mejora "requerimientos de mejora de vivienda" 
label define req_mejora 0 "sin requerimiento de mejora" 1 "requerimiento de mejora"
label values req_mejora req_mejora

gen tip_req_mejora=.
replace tip_req_mejora=1 if (pco1==1) & (req_mejora==1) & (ind_estado<3)  & (ind_cal_glob==2)
replace tip_req_mejora=2 if (pco1==1) & (req_mejora==1) & (ind_estado==3) & (ind_cal_glob<2)
replace tip_req_mejora=3 if (pco1==1) & (req_mejora==1) & (ind_estado==3) & (ind_cal_glob==2)

label var tip_req_mejora "tipo de requerimiento de mejora"
label define tip_req_mejora								///
1 "materialidad recuperable y estado bueno o regular"	///
2 "materialidad aceptable y estado malo"				///
3 "materialidad recuperable y estado malo"
label values tip_req_mejora tip_req_mejora

***************************************************
*******Requerimientos por viviendas irrecuperables*
***************************************************

capture drop req_viv_irr

gen req_viv_irr=.
replace req_viv_irr=1 if (pco1==1) & (hog_prin==1) & (ind_cal_glob==3)
label var req_viv_irr "Déficit cuantitativo: requerimientos por vivienda irrecuperable"
label define req_viv_irr 1 "hogares principales en vivienda irrecuperable"
label values req_viv_irr req_viv_irr
bys id_hogar: egen hog_req_viv_irr=max(req_viv_irr)

***************************************************
*******Requerimiento de ampliación*****************
***************************************************
gen req_amplia=.
replace req_amplia=0 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_hacina==1)
replace req_amplia=0 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_hacina>=2) & (ind_hacina<99) & (allega_int==1)
replace req_amplia=1 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_hacina>=2) & (ind_hacina<99) & (allega_int==0)

label var req_amplia "requerimientos de ampliación de vivienda" 
label define req_amplia 0 "sin requerimiento de ampliación" 1 "requerimiento de ampliación"
label values req_amplia req_amplia

***************************************************
*******Requerimiento vivienda no ampliable*********
***************************************************
*Requerimientos por hogares en viviendas no ampliables
gen pieza_departamento=(inlist(v1,4,5,6))
label variable pieza_departamento "Vivienda no ampliable (departamento o pieza)"
label define pieza_departamento 0 "Es de otro tipo", modify
label define pieza_departamento 1 "Es departamento o pieza", modify
label values pieza_departamento pieza_departamento

**Departamento o pieza no ampliable
gen req_vivnoamp_pza_dpto=0
replace req_vivnoamp_pza_dpto=1 if pco1==1 & hog_prin==1 & inlist(ind_hacina,2,3,4) & allega_int==0 & inlist(ind_cal_glob,1,2) & pieza_departamento==1

**Arriendo no ampliable
capture drop ten_viv
recode v13 (1=1)(2=2)(3=3)(9 10 11 .=4),gen(ten_viv)
gen req_vivnoamp_arriendo=0
replace req_vivnoamp_arriendo=1 if pco1==1 & hog_prin==1 & inlist(ind_hacina,2,3,4) & allega_int==0 & inlist(ind_cal_glob,1,2) & ten_viv==2 & req_vivnoamp_pza_dpto==0

**Vivienda no ampliable
gen req_vivnoamp=0
replace req_vivnoamp=1 if req_vivnoamp_pza_dpto==1 | req_vivnoamp_arriendo==1
label var req_vivnoamp "Déficit Cuantitativo: requerimientos por viviendas no ampliables hacinadas (hogar principal hacinado) no ampliables"
label define req_vivnoamp 0 "sin requerimiento por vivienda no ampliable", modify
label define req_vivnoamp 1 "con requerimiento por vivienda no ampliable", modify
label values req_vivnoamp req_vivnoamp
bys id_hogar: egen hog_req_vivnoamp=max(req_vivnoamp)

***************************************************
*******Requerimiento de ampliación ajustado********
***************************************************
	gen req_amplia_ajustado=.
replace req_amplia_ajustado=0 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_hacina==1)
replace req_amplia_ajustado=0 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_hacina>=2) & (ind_hacina<99) & (allega_int==1)
replace req_amplia_ajustado=1 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_hacina>=2) & (ind_hacina<99) & (allega_int==0)
replace req_amplia_ajustado=0 if req_vivnoamp==1

label var req_amplia_ajustado "requerimientos de ampliación de vivienda ajustado" 
label define req_amplia_ajustado 0 "sin requerimiento de ampliación", modify
label define req_amplia_ajustado 1 "requerimiento de ampliación", modify
label values req_amplia_ajustado req_amplia_ajustado

***************************************************
*******Requerimiento de saneamiento****************
***************************************************

	gen req_sanea=.
replace req_sanea=0 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_san==1)
replace req_sanea=1 if (pco1==1) & (hog_prin==1) & (ind_cal_glob<3) & (ind_san==2)

label var req_sanea "requerimientos de saneamiento de vivienda" 
label define req_sanea 0 "sin requerimiento de saneamiento" 1 "requerimiento de saneamiento"
label values req_sanea req_sanea

egen def_cuali=rowtotal(req_mejora req_amplia req_sanea)

***************************************************
*******Déficit cualitativo ajustado****************
***************************************************
egen def_cuali_ajustado=rowtotal(req_mejora req_amplia_ajustado req_sanea) if (req_viv_irr !=1 | req_viv_irr ==.)

label var def_cuali "recuento déficit cualitativo a nivel de hogares"
label define def_cuali 0 "sin requerimientos", modify
label define def_cuali 1 "1 requerimiento", modify
label define def_cuali 2 "2 requerimientos", modify
label define def_cuali 3 "3 requerimientos", modify
label values def_cuali def_cuali

label var def_cuali_ajustado "recuento déficit cualitativo a nivel de hogares"
label define def_cuali_ajustado 0 "sin requerimientos", modify
label define def_cuali_ajustado 1 "1 requerimiento", modify
label define def_cuali_ajustado 2 "2 requerimientos", modify
label define def_cuali_ajustado 3 "3 requerimientos", modify
label values def_cuali_ajustado def_cuali_ajustado

recode def_cuali (0=0)(1/3=1),g(def_cuali_tot)
label define def_cuali_tot 0 "sin requerimientos", modify
label define def_cuali_tot 1 "1 o más requerimientos", modify
label values def_cuali_tot def_cuali_tot

***************************************************
*******Déficit cualitativo total ajustado**********
***************************************************
recode def_cuali_ajustado (0=0)(1/3=1),g(def_cuali_tot_ajustado)
label define def_cuali_tot_ajustado 0 "sin requerimientos", modify
label define def_cuali_tot_ajustado 1 "1 o más requerimientos", modify
label values def_cuali_tot_ajustado def_cuali_tot_ajustado

*Requerimientos por hogares allegados
gen req_hog_all=.
replace req_hog_all=n_hog_allega if (pco1==1) & (hog_prin==1)
label var req_hog_all "Déficit cuantitativo: requerimientos por hogares allegados"
label values req_hog_all req_hog_all
bys id_hogar: egen hog_req_hog_all=sum(req_hog_all)
*Requerimientos por núcleos allegados hacinados
gen req_nuc_all=.
replace req_nuc_all=1 if pco2==1 & nuc_prin==2 & ind_hacina>=2 & ind_hacina<99
label var req_nuc_all "Déficit cuantitativo: requerimientos por núcleos allegados y hacinados"
label define req_nuc_all 1 "núcleos allegados y hacinados"
label values req_nuc_all req_nuc_all
bys id_hogar: egen hog_req_nuc_all=sum(req_nuc_all)

*Déficit cuantitativo antigua y nueva metodología (ajustado)
egen def_cuanti=rowtotal(hog_req_viv_irr hog_req_hog_all hog_req_nuc_all)
egen def_cuanti_ajustado=rowtotal(hog_req_viv_irr hog_req_hog_all hog_req_nuc_all hog_req_vivnoamp)

recode def_cuanti_ajustado (0=0)(1/9=1),g(def_cuanti_tot_ajustado)
label define def_cuanti_tot_ajustado 0 "sin requerimientos", modify
label define def_cuanti_tot_ajustado 1 "1 o más requerimientos", modify
label values def_cuanti_tot_ajustado def_cuanti_tot_ajustado

*Carencia
capture drop d_defcuali
gen d_defcuali = 0 if def_cuali_tot_ajustado == 0 & pob_defcuali == 1
replace d_defcuali = 1 if def_cuali_tot_ajustado == 1 & pob_defcuali == 1  
replace d_defcuali = 9 if d_defcuali == . & pob_defcuali == 1 

capture drop d_defcuanti
gen d_defcuanti = 0 if def_cuanti_tot_ajustado == 0 & pob_defcuanti == 1
replace d_defcuanti = 1 if def_cuanti_tot_ajustado == 1 & pob_defcuanti == 1  
replace d_defcuanti = 9 if d_defcuanti == . & pob_defcuanti == 1 

********************************************************************************
/* Indicador 15: Accesibilidad

Un hogar es carente en accesibilidad si:
	• los hogares que no tienen integrantes ocupados y no disponen de alguno de 
	los siguientes equipamientos básicos a una distancia cercana desde su 
	vivienda: i. Servicio de transporte público (paradero, estación) a menos 
	de 8 cuadras o 1 km de su vivienda; ii. Centros educacionales (colegio o 
	jardín infantil) a menos de 20 cuadras o 2,5 km de su vivienda; y 
	iii. Centros de salud (atención primaria o nivel superior a menos de 20 
	cuadras o 2,5 km de su vivienda; o,
	• Los hogares tienecon uno o más integrantes ocupados, que carecen de alguno 
	de los equipamientos básicos antes mencionados a una distancia cercana 
	desde su vivienda y alguna de las personas ocupadas, demoran diariamente 
	una hora o más en promedio en llegar a su lugar de trabajo principal 
	(usando transporte público o transporte no motorizado).


*/
********************************************************************************


********************
/* EQUIPAMIENTO: 

No dispone de alguno de estos tres equipamientos mínimos: 

a. Transporte público.
b. Centro educacional. 
c. Centro de salud

*/
********************

*Población de referencia
capture drop pob_equipo
gen pob_equipo=(numper!=.)
replace pob_equipo=8 if numper==.

*Servicio de transporte público
capture drop d_eqtr
gen d_eqtr=.
replace d_eqtr=1 if v35a==2
replace d_eqtr=0 if v35a==1
replace d_eqtr=. if inlist(v35a,-88,88)

*Servicio de centro educacional
capture drop d_eqce
gen d_eqce=.
replace d_eqce=1 if v35b==2
replace d_eqce=0 if v35b==1
replace d_eqce=. if inlist(v35b,-88,88)

*Servicio de centro de salud
capture drop d_eqcs
gen d_eqcs=.
replace d_eqcs=1 if v35c==2
replace d_eqcs=0 if v35c==1
replace d_eqcs=. if inlist(v35c,-88,88)

*Acceso a equipos
capture drop d_equipo
gen d_equipo=.
replace d_equipo=0 if (d_eqtr==0 & d_eqce==0 & d_eqcs==0) & pob_equipo==1
replace d_equipo=1 if (d_eqtr==1 | d_eqce==1 | d_eqcs==1) & pob_equipo==1
replace d_equipo=9 if d_equipo==. & pob_equipo==1


********************
/* TIEMPO DE TRASLADO: 

Tiempo de traslado al trabajo en transporte público o particular 
no motorizado igual o mayor a 60 min.

*/
********************

* Población de referencia
capture drop pob_tiempo
g pob_tiempo=(activ==1)
replace pob_tiempo=8 if activ==.

* Privación a nivel individual
capture drop d_tiempo
gen d_tiempo=0 if (o28a_hr<1 & o28a_hr!=-8 & o28a_hr!=.) & pob_tiempo==1
replace d_tiempo=1 if (o28a_hr>=1 & o28a_hr!=-8 & o28a_hr!=.) & pob_tiempo==1
replace d_tiempo=. if o28a_hr==. & pob_tiempo==1
replace d_tiempo=. if o28a_hr==-8 & pob_tiempo==1

*Medio de transporte
replace d_tiempo=0 if (o28c==2 | o28c==6) & d_tiempo==1 & pob_tiempo==1
replace d_tiempo=. if (o28c==.) & d_tiempo==1 & pob_tiempo==1
replace d_tiempo=. if (o28c==-88) & d_tiempo==1 & pob_tiempo==1

*CARENCIA EN TIEMPO DE TRASLADO POR HOGAR
capture drop oc hh_oc d_hh_tiempo
gen oc = (activ==1)
bys folio: egen hh_oc=max(oc)
bys folio: egen d_hh_tiempo=max(d_tiempo) if hh_oc==1

********************
/*
Carencia de Equipamiento Comunitario:
*/
********************
capture drop pob_accesi
gen pob_accesi = (numper!=.)
replace pob_accesi = 8 if numper == .

capture drop d_accesi
gen d_accesi=.
replace d_accesi=1 if (d_hh_tiempo==1 & d_equipo==1) & hh_oc==1
replace d_accesi=0 if (d_hh_tiempo==0 | d_equipo==0) & hh_oc==1
replace d_accesi=1 if (d_equipo==1) & hh_oc==0
replace d_accesi=0 if (d_equipo==0) & hh_oc==0
replace d_accesi=9 if d_accesi==.



********************************************************************************
/* Indicador 16: Contaminación

Un hogar es carente en contaminación si, durante los últimos 12 meses:
	• El hogar ha vivido o presenciado “siempre” o “muchas veces” a lo menos 
	dos problemas de contaminación medioambiental en su área de residencia (a 
	una distancia de no más de 15 minutos caminando desde su vivienda), 
	incluyendo los siguientes: i) contaminación del aire y/o malos olores; 
	ii) contaminación en ríos, canales, esteros, lagos, tranques y embalses; 
	iii) contaminación del agua proveniente de la red pública; y/o, 
	iv) acumulación de basura en calles, caminos, veredas o espacios públicos.


*/
********************************************************************************
capture drop pob_medio
*población de referencia
gen pob_medio = (numper!= .)
*Situaciones de contaminación:

*Aire
capture drop cont_aire
gen cont_aire=.
replace cont_aire=1 if (inlist(v37b,3,4))
replace cont_aire=0 if (inlist(v37b,1,2))
label variable cont_aire "Contaminación del aire"

*Contaminación de ríos, canales
capture drop cont_rios
gen cont_rios=.
replace cont_rios=1 if (inlist(v37c,3,4))
replace cont_rios=0 if (inlist(v37c,1,2))
label variable cont_rios "Contaminación de ríos, canales, esteros, lagos, tranques y embalses"

*Contaminación de agua de red pública
capture drop cont_agua
gen cont_agua=.
replace cont_agua=1 if (inlist(v37d,3,4))
replace cont_agua=0 if (inlist(v37d,1,2))
label variable cont_agua "Contaminación de agua de red pública"

*Contaminación por basura
capture drop cont_basura
gen cont_basura=.
replace cont_basura=1 if (inlist(v37f,3,4))
replace cont_basura=0 if (inlist(v37f,1,2))
label variable cont_basura "Acumulación de basura en calles, caminos, veredas o espacios públicos"
label define contaminacion 0 "Nunca, Pocas" 1 "Muchas veces o Siempre", modify

label values cont_aire contaminacion
label values cont_rios contaminacion
label values cont_agua contaminacion
label values cont_basura contaminacion

*Suma de situaciones de contaminación
capture drop sum_con
egen sum_con=rowtotal(cont_aire cont_rios cont_agua cont_basura)

* Privación a nivel individual
capture drop d_medio
gen d_medio=.
replace d_medio=1 if (sum_con>=2) & pob_medio==1
replace d_medio=0 if (sum_con<2) & pob_medio==1
replace d_medio=9 if cont_aire==. & cont_rios==. & cont_agua==. & cont_basura==.

}


********************************************************************************
***************************REDES Y COHESIÓN*************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 17: Apoyo
Un hogar es carente en apoyo si:
	• Ninguna persona integrante del hogar conoce a alguien fuera del hogar 
	que pueda prestar apoyo en alguna de las siguientes situaciones: 1. Ayuda 
	en el cuidado en caso de enfermedad de algún miembro del hogar; 2. Facilitar 
	un vehículo si el hogar lo necesita; 3. Prestar dinero al hogar en caso de 
	emergencia; 4. Ayudar al hogar a resolver consultas o realizar trámites 
	legales o financieros; 5. Ayudar al hogar en el uso de tecnologías; 
	6. Ayudar con reparaciones del hogar; 7. Ayudar a conseguir un trabajo a 
	algún miembro del hogar; 8. Aconsejar en caso problemas personales o 
	familiares.

*/
********************************************************************************

*Poblacion de referencia
capture drop pob_apoyo
gen pob_apoyo=(numper!=.)
replace pob_apoyo=8 if numper==.

* Se identifican las categorías en las cuales se considera carente para cada situación
*Cuidado en caso de enfermedad en algun miembro del hogar
gen apoyo_a=.
replace apoyo_a=0 if r7a==4 
replace apoyo_a=0 if r7a==2
replace apoyo_a=1 if r7a==1
replace apoyo_a=1 if r7a==3

*Facilitar vehiculo
gen apoyo_c=.
replace apoyo_c=0 if r7c==4
replace apoyo_c=0 if r7c==2
replace apoyo_c=1 if r7c==1
replace apoyo_c=1 if r7c==3

*Prestar dinero
gen apoyo_d=.
replace apoyo_d=0 if r7d==4
replace apoyo_d=0 if r7d==2
replace apoyo_d=1 if r7d==1
replace apoyo_d=1 if r7d==3

*Hacer tramites
gen apoyo_e=.
replace apoyo_e=0 if r7e==4
replace apoyo_e=0 if r7e==2
replace apoyo_e=1 if r7e==1
replace apoyo_e=1 if r7e==3

*Ayudar en uso de teconologias
gen apoyo_f=.
replace apoyo_f=0 if r7f==4
replace apoyo_f=0 if r7f==2
replace apoyo_f=1 if r7f==1
replace apoyo_f=1 if r7f==3

*Reparaciones en el hogar
gen apoyo_g=.
replace apoyo_g=0 if r7g==4
replace apoyo_g=0 if r7g==2
replace apoyo_g=1 if r7g==1
replace apoyo_g=1 if r7g==3

*Conseguir trabajo
gen apoyo_h=.
replace apoyo_h=0 if r7h==4
replace apoyo_h=0 if r7h==2
replace apoyo_h=1 if r7h==1
replace apoyo_h=1 if r7h==3

*Aconsejar
gen apoyo_i=.
replace apoyo_i=0 if r7i==4
replace apoyo_i=0 if r7i==2
replace apoyo_i=1 if r7i==1
replace apoyo_i=1 if r7i==3

label define apoyo_lbl_mds 0 "No conoce fuera del hogar" 1 "Conoce fuera del hogar"
label value apoyo_a apoyo_lbl_mds

label value apoyo_c apoyo_lbl_mds
label value apoyo_d apoyo_lbl_mds
label value apoyo_e apoyo_lbl_mds
label value apoyo_f apoyo_lbl_mds
label value apoyo_g apoyo_lbl_mds
label value apoyo_h apoyo_lbl_mds
label value apoyo_i apoyo_lbl_mds
label variable apoyo_a "Enfermedad"
label variable apoyo_c "Vehiculo"
label variable apoyo_d "Dinero"
label variable apoyo_e "Trámites"
label variable apoyo_f "Tecnologías"
label variable apoyo_g "Reparaciones"
label variable apoyo_h "Trabajo"
label variable apoyo_i "Problemas personales"

*** Agregacion por hogar
local apoyo a c d e f g h i
foreach var in `apoyo' {
	bys folio: egen hapoyo_`var'=max(apoyo_`var')
}

* Genera suma que permite identificas carencia
egen c_apoyo=rsum(hapoyo_a hapoyo_c hapoyo_d hapoyo_e hapoyo_f hapoyo_g hapoyo_h hapoyo_i)
egen c_apoyo_miss=rmiss(hapoyo_a hapoyo_c hapoyo_d hapoyo_e hapoyo_f hapoyo_g hapoyo_h hapoyo_i)
replace c_apoyo=. if c_apoyo_miss>=1 & c_apoyo==0
label variable c_apoyo "Indicador apoyos sociales"
label variable c_apoyo_miss "Contador de variables perdidas en apoyos"


* Privación si para todas las situaciones no cuenta con alguna persona a fuera del hogar.
gen d_apoyo=.
replace d_apoyo=1 if c_apoyo==0 & pob_apoyo==1 & c_apoyo!=. & c_apoyo_miss==0
replace d_apoyo=0 if (c_apoyo>=1) & pob_apoyo==1 & c_apoyo!=.
replace d_apoyo=9 if d_apoyo==. & pob_apoyo==1


********************************************************************************
/* Indicador 18: Trato igualitario

Un hogar es carente en trato igualitario si:
	• Alguna persona integrante ha sido tratada injustamente o discriminada 
	fuera del hogar, en los últimos 12 meses, por alguna de las siguientes 
	razones: 1. su nivel socioeconómico; 2. ser hombre/mujer; 3. su estado 
	civil; 4. su ropa; 5. su color de piel; 6. ser extranjero; 7. su edad; 
	8. su orientación sexual o identidad de género; 9. tener tatuajes, piercing, 
	perforaciones o expansiones; 10. su apariencia física; 11. sus creencias o 
	religión; 12. su ideología u opinión política; 13. participar o no en 
	sindicatos u organizaciones gremiales; 14. el lugar donde vive; 15. el 
	establecimiento donde estudió; 16. pertenecer a un pueblo indígena; 17. su 
	condición de salud o discapacidad; 18. su nivel educacional; o 19. su 
	trabajo u ocupación.

*/
********************************************************************************
*Poblacion de referencia
capture drop pob_tsocial d_tsocial
gen pob_tsocial=(numper!=.)
replace pob_tsocial=8 if numper==.

* Privación a nivel individual
gen d_tsocial=.
replace d_tsocial=1 if ( r9a==1 | r9b==1 | r9c==1 | r9d==1 | r9e==1 | r9f==1 | r9g==1 | r9h==1 | r9i==1 | r9j==1 | r9k==1 | r9l==1 | r9m==1 | r9n==1 | r9o==1 | r9p==1 | r9q==1 | r9r==1 | r9u==1 ) & pob_tsocial==1

replace d_tsocial=0 if (r9s==1) & d_tsocial==. & pob_tsocial==1
replace d_tsocial=0 if (r9t==1) & d_tsocial==. & pob_tsocial==1


********************************************************************************
/* Indicador 19: Conectividad digital

Un hogar es carente en conectividad digital si:
	• No cuentan con banda ancha fija, banda ancha móvil (módem, Wifi o USB), 
tablet u otro dispositivo que permita conexión a Internet (excluido el 
teléfono móvil a excepción de hogares unipersonales), o internet satelital.



*/
********************************************************************************
*Poblacion de referencia
capture drop pob_conec 

gen pob_conec = (numper != .)
replace pob_conec=8 if numper==.

*Carencia individual
capture drop d_conec
gen d_conec = 0
replace d_conec = 1 if (r17a == 2 & r17b == 2 & r17d == 2 & r17e == 2) & pob_conec == 1
replace d_conec = 0 if d_conec == 1 & inlist(r18,8,10) & pob_conec == 1
replace d_conec = 9 if inlist(r18,-88,-99) & pob_conec == 1
replace d_conec = 0 if d_conec == 1 & r17c == 1 & numper == 1

********************************************************************************
/* Indicador 20: Seguridad

Un hogar es carente en seguridad si:
	• Declara que alguna de las personas integrantes ha vivido o presenciado 
“siempre” o “muchas veces” durante el último mes, a lo menos una de las 
siguientes situaciones: i) Tráfico de drogas; ii) Balaceras o disparos; 
iii) Peleas o amenazas.

*/
********************************************************************************

capture drop trafico
g trafico=.
replace trafico=1 if inlist(v36c,3,4)
replace trafico=0 if inlist(v36c,1,2)

capture drop balacera
g balacera=.
replace balacera=1 if inlist(v36e,3,4)
replace balacera=0 if inlist(v36e,1,2)

capture drop peleas
g peleas=.
replace peleas=1 if inlist(v36d,3,4)
replace peleas=0 if inlist(v36d,1,2)

label define seguridad 0 "Nunca o Pocas veces" 1 "Muchas veces o Siempre"
label values trafico seguridad
label values balacera seguridad
label values peleas seguridad

*Población de referencia
gen pob_seg=(numper!=.)
replace pob_seg=8 if pob_seg==.
*Indicador (alguna de las situaciones críticas ocurre siempre)
gen d_seg=.
replace d_seg=0 if pob_seg==1
replace d_seg=1 if (trafico==1 | balacera==1| peleas==1) & pob_seg==1
replace d_seg=9 if d_seg==. & pob_seg==1

*Algo que no estaba en la prog historica: Llevamos a missing si en ambas situaciones responde que no sabe
replace d_seg = 9 if balacera == . & trafico == . & peleas == .

}


********************************************************************************
**********************AGREGACIÓN A NIVEL DE HOGARES*****************************
{/*****************************************************************************/

local miss asis rez esc ape acc ali contprev dpf actsub inf jub cui defcuali defcuanti accesi medio apoyo tsocial conec seg


foreach var in `miss' {
	g pob_`var'_c=pob_`var'
	replace pob_`var'_c=0 if (pob_`var'_c==8 | pob_`var'_c==.)
	g d_`var'_c=d_`var'
	replace d_`var'_c=. if (d_`var'_c==9 | d_`var'_c==.)
}

local pob asis rez esc ape acc ali contprev dpf actsub inf jub cui defcuali defcuanti accesi medio apoyo tsocial conec seg

foreach var in `pob' {
	bys folio: egen hh_pob_`var'=max(pob_`var'_c)
	bys folio: egen hh_d_`var'=max(d_`var'_c)
	replace hh_d_`var'=0 if hh_pob_`var'==0
	
	label var d_`var'_c "Individuos carentes en `var' de población relevante"
}


label var hh_d_asis "hogar carente en asistencia escolar"
label var hh_d_rez "hogar carente en rezago escolar"
label var hh_d_esc "hogar carente en escolaridad"
label var hh_d_ape "hogar carente en aprendizaje escolar en el establecimiento"

label var hh_d_acc "hogar carente en atención de salud"
label var hh_d_ali "hogar carente en acceso a alimentos"
label var hh_d_contprev "hogar carente en acceso a salud preventiva"
label var hh_d_dpf "hogar carente en apoyo en cuidado de personas con dependencia funcional"

label var hh_d_actsub "hogar carente en ocupación y subempleo"
label var hh_d_inf "hogar carente en informalidad"
label var hh_d_jub "hogar carente en jubilacion"
label var hh_d_cui "hogar carente en participación por cuidados"

label var hh_d_defcuali "hogar carente en deficit cualitativo"
label var hh_d_defcuanti "hogar carente en deficit cuantitativo"
label var hh_d_accesi "hogar carente en equipamiento accesibilidad"
label var hh_d_medio "hogar carente en contaminación"

label var hh_d_apoyo "hogar carente en apoyo"
label var hh_d_tsocial "hogar carente en trato igualitario"
label var hh_d_conec "hogar carente en conectividad digital"
label var hh_d_seg "hogar carente en seguridad"

foreach var in `pob' {
	label define hh_d_`var' 0 "No carente" 1 "Carente", modify
	label values hh_d_`var' hh_d_`var'
}

}


********************************************************************************
**************************INDICADOR SINTETICO***********************************
{/*****************************************************************************/
*PONDERACIÓN
local pob asis rez esc ape acc ali contprev dpf actsub inf jub cui defcuali defcuanti accesi medio apoyo tsocial conec seg
foreach var in `pob' {
* Peso para cada indicador
capture drop pp_`var'
gen pp_`var'= 0.20/4
* Carencia ponderada
capture drop  w_hh_d_`var'
gen w_hh_d_`var'= hh_d_`var'* pp_`var'
*label var hh_d_`var' "Hogar carente en `var'",
label var pp_`var' "Ponderador `var'"
label var w_hh_d_`var' "Carencia ponderada de `var'"
}

*5.2 Vector de conteo
* El siguiente comando suma todos los indicadores ponderados
capture drop ci
egen float ci=rsum(w_hh_d_*)
label var ci "Vector de Conteo"

* Se identifican hogares excluidos por no contar con información para todos los indicadores
capture drop missing
egen missing=rmiss(w_hh_d_*)
capture drop dmissing
gen dmissing=(missing!=0)
label var missing "Conteo de Missing"

* 5.3. Identificación y Agregación
capture drop h_k
gen h_k=(ci>=0.25) if dmissing!=1
label var h_k "situación de pobreza multidimensional"
capture drop a_k
gen a_k=ci if h_k==1
label var a_k "Intensidad de la pobreza multidimensional"
capture drop m_k
gen m_k=0 if dmissing!=1
replace m_k=ci if h_k==1
label var m_k "Tasa de recuento ajustado de pobreza multidimensional"

capture drop pobreza_multi
rename h_k pobreza_multi
label define pobreza_multi 0 "Hogar fuera de pobreza multidimensional" 1 "Hogar en pobreza multidimensional",modify
label values pobreza_multi pobreza_multi

}

